#!/usr/bin/env python3
from textual.app import App, ComposeResult
from textual.containers import VerticalScroll, Vertical
from textual.widgets import Footer, Header, TabbedContent, TabPane, Static, LoadingIndicator #, Sparkline
from textual_plotext import PlotextPlot
from textual import work  #, on
from rich.text import Text
from rich_gradient.text import Text as Gradiant
import subprocess
import requests #httpx
import urllib.parse
from configparser import ConfigParser
import datetime
import tzlocal
import pytz
import getpass
import os

'''
     ...     ..         .                                             .
  .=*8888x <"?88h.     @88>    ..                                    @88>
 X>  '8888H" '8888     %8P    @L                 u.      .u    .     %8P
'88h. `8888    888      .    9888i   .dL   ...ue888b   .d88B :@8c     .
'8888 '8888    ;88>   .@88u  `Y888k:*888.  888R Y888r ="8888f8888r  .@88u
 `888 '8888.xH888x.  ''888E`   888E  888I  888R  888>   4888>'88"  ''888E`
   X" :88*~  `*8888>   888E    888E  888I  888R  888>   4888> '      888E
 ~"   !"`      "888>   888E    888E  888I  888R  888>   4888>        888E
  .H8888h.      ?88    888E    888E  888I u8888cJ888   .d888L .+     888E
 :"^"88888h.    '!     888&   x888N><888'  "*888*P"    ^"8888*"      888&
 ^    "88888hx.+"      R888"   "88"  888     'Y"          "Y"        R888"
        ^"**""          ""           88F                              ""
                                    98"
                                  ./"
                                 ~`

Biyori: Textual TUI application serving US weather and alerts.
Author: C. Nichols <mohawke@gmail.com>

This is a work in progress. I am learning as I go and will update when I can. When I
feel it's ready, I might push to GitHub.

Only supports US weather.
Designed to be used as a widget in Wave terminal, but runs better in most terminals. :(
Retrieves weather data from Open Meteo and NWS. Both are free services so be kind with the
amount of requests you ask of them. Refreshing every 15 to > 60 minutes should be fine.

Dependencies:
    ConfigParser
    textual
    plotextplot
    requests
    rich-gradient
    cheap_repr
    snoop
    tzlocal
    pytz

Config:
When you run the program for the first time the config will get created in .config in
your home directory. This folder is hidden so turn on show hidden files if you don't see
it.

The folder is .config/biyori and the file is biyori.conf. I thought about creating a popup form
to set these, but it's a set once and forget so...

* The defaults set in conf. file (change what you need.):

[DEFAULT]
location = 40.036282,-83.000762
state = Ohio
temperature_unit = fahrenheit
wind_speed_unit = mph
precipitation_unit = inch
refresh_rate = 15
'''

# ASCII Icons
sun_clear = '''[yellow]
              ""
              ""
     "        ""        "
        "  SSSSSSSS  "
          SSSSSSSSSS
        SSSSSSSSSSSSSS
====== SSSSSSSSSSSSSSSS ======
        SSSSSSSSSSSSSS
          SSSSSSSSSS
           SSSSSSSS
        "     ""    "
     "        ""       "
              ""
[/yellow]'''

partly_sunny = Gradiant('''

             "
             "
    "        "       "
        " ,sssss, "
         s       s
 """""" s        ▒▒▒▒""""
         ▒▒▒▒s▒▒pppp▒▒
      ▒▒▒▒▒ppp▒pp▒▒▒▒pp▒▒▒
    ▒▒▒▒▒ppp▒▒▒▒pppp▒▒▒▒▒▒▒
         ▒▒▒pppp▒▒▒▒▒
            ▒▒▒▒
''', colors=["#FFFF00","#FFFF8F","#FFFFFF"])

thunderstorm_rain = Gradiant("""

             ▒▒▒▒▒
       ▒▒▒ ▒▒tttt▒▒▒
     ▒▒ttt▒ttt▒▒▒▒tt▒▒▒
   ▒▒▒▒tttt▒▒▒▒tttt▒▒▒▒▒▒
      ▒▒▒tttt▒▒▒▒▒
         ▒▒▒  /     '
     '     / /_   '  '
      '   /_  /    '
           / /      '
          //     '
          /       '
""", colors=["#8B8000","#DAA520","#FAFA33"])

thunderstorm_hail = Gradiant("""

             ▒▒▒▒▒
       ▒▒▒ ▒▒tttt▒▒▒
     ▒▒ttt▒ttt▒▒▒▒tt▒▒▒
   ▒▒▒▒tttt▒▒▒▒tttt▒▒▒▒▒▒
      ▒▒▒tttt▒▒▒▒▒
         ▒▒▒  /    \\
    \\     / /_  \\   '
      @   /_  /    @
    \\    / /       ' \\
       @   //    \\     @
          /        @     '
""", colors=["#8B8000","#DAA520","#FAFA33"])

thunderstorm = Gradiant("""

             ▒▒▒▒▒
       ▒▒▒ ▒▒tttt▒▒▒
     ▒▒ttt▒ttt▒▒▒▒tt▒▒▒
   ▒▒▒▒tttt▒▒▒▒tttt▒▒▒▒▒▒
      ▒▒▒tttt▒▒▒▒▒
         ▒▒▒  /
           / /_
          /_  /
           / /
          //
          /
""", colors=["#8B8000","#DAA520","#DAA520"])

light_rain = Gradiant('''

             ▒▒▒▒▒
       ▒▒▒▒▒▒rrrrr▒▒
     ▒▒rrr▒rr▒▒▒▒rr▒▒▒
   ▒▒▒▒rrr▒▒▒▒rrrr▒▒▒▒▒▒
      ▒▒▒rrrr▒▒▒▒▒
         ▒▒▒▒      '    '
     '        '     '    '
      '    '   '          '
            '         '
        '        '     '    '
         '    '   '
''', colors=["#A9A9A9","#9FAECC","#6E9EFF"])

rain = Gradiant('''

             ▒▒▒▒▒
       ▒▒▒▒▒▒rrrrr▒▒
     ▒▒rrr▒rr▒▒▒▒rr▒▒▒
   ▒▒▒▒rrr▒▒▒▒rrrr▒▒▒▒▒▒
      ▒▒▒rrrr▒▒▒▒▒
         ▒▒▒▒      \\     \\
     \\        \\       \\
          \\
                 \\   \\
         \\    \\
                 \\
''', colors=["#A9A9A9","#9FAECC","#6E9EFF"])

heavy_rain = Gradiant('''

             ▒▒▒▒▒
       ▒▒▒▒▒▒rrrrr▒▒
     ▒▒rrr▒rr▒▒▒▒rr▒▒▒
   ▒▒▒▒rrr▒▒▒▒rrrr▒▒▒▒▒▒
      ▒▒▒rrrr▒▒▒▒▒
         ▒▒▒▒     \\    \\
    \\       \\      ' \\  '
      '   \\   '         '
            '    \\  \\
        \\   \\    '   '
          '    '  \\
''', colors=["#A9A9A9","#9FAECC","#6E9EFF"])

icy_rain = Gradiant('''

             ▒▒▒▒▒
       ▒▒▒▒▒▒iiiii▒▒
     ▒▒iii▒ii▒▒▒▒ii▒▒▒
   ▒▒▒▒iii▒▒▒▒iiii▒▒▒▒▒▒
      ▒▒▒iiii▒▒▒▒▒
         ▒▒▒▒     \\    \\
    \\       \\     * \\  *
      *   \\   *        *
            *    \\  \\
        \\   \\    *   *
          *    *  \\
''', colors=["#A9A9A9","#9FAECC","#6E9EFF"])

heavy_snow = Gradiant('''

             ▒▒▒▒▒
       ▒▒▒▒▒▒ssss▒▒
     ▒▒sss▒ss▒▒▒▒ss▒▒▒
   ▒▒▒▒sss▒▒▒▒ssss▒▒▒▒▒▒
      ▒▒▒ssss▒▒▒▒▒
         ▒▒▒▒        *
     *         +     +   *
      +    *     *         +
            +     *     *   *
      *          +
         +    *           *
''', colors=["#A9A9A9","#9FAECC","#6E9EFF"])


snow = '''[white]

               !
               !
          .    #    .
          \\  ###  /
          # \\ " /  #
 --~ ##  #  (  o  )  # ## ~--
          #  / "\\  #
           /  ### \\
          .    #    .
               !
               !
[/white]'''

moon_clear = Gradiant('''
      .         -=m:       .        .      
                   mmm:
.                    mmm:      +
                      mmm~
                      :mmm}                . 
                      :mmmm           .
                      :mmm}   *             
                      mmm~
*                    mmm:
                   mmm:    .       +
                -=m:      .               .
''', colors=["#DFE8FC","#CDDDFF"])

partly_cloudy = Gradiant('''

                _-~~~-_
              m         m      *
             /      ()   \\    .
            m             m
            m   o         m
             \\        @  /
   *          m▒▒▒▒c▒▒ccccc▒▒           o
        ▒▒▒▒▒cc▒cc▒▒▒▒cc▒▒▒▒▒▒▒▒▒
     ▒▒▒▒▒▒ccc▒▒▒▒▒ccc▒▒▒▒cccc▒▒▒▒▒▒
                  ▒▒▒▒▒▒
       +                    ▒▒▒▒▒
                         ▒▒▒ccccc▒▒▒▒▒
''', colors=["#FFFFFF","#D4EDFA","#D4EDFA"])

cloudy = Gradiant('''
                ▒▒▒
             ▒▒▒▒c▒▒▒▒▒▒▒
        ▒▒▒▒▒cc▒cc▒▒▒▒cc▒▒▒▒▒▒
     ▒▒▒▒▒▒ccc▒▒▒▒▒ccc▒▒▒▒cccc▒▒▒
           ▒▒▒▒    ▒▒▒▒▒▒
                             ▒▒▒▒▒
             ▒ ▒▒▒▒        ▒▒▒cccc▒▒▒
         ▒▒▒▒c▒▒cc▒▒▒▒▒▒      ▒▒▒▒
    ▒▒▒▒▒cc▒cc▒▒▒▒cc▒▒▒▒▒▒▒▒
  ▒▒▒▒▒▒ccc▒▒▒▒▒ccc▒▒▒▒cccc▒▒▒▒▒
       ▒▒▒▒cc▒▒▒▒▒▒
          ▒▒▒▒
''', colors=["#F8F8F8","#A0A0A0"])

windy = Gradiant('''

                 ←←↖ ↖←←
               ←←       ←
               ←←
      ← ↖ ←       ↖ ←←←←←←←←←←←←←←←←←←←←←←←
    ←←      ←        ←←←←←←←←←←←←←←←←←←←←←←←←←
    ↖              ____________________
      ↖ ←←←←←←←←←←←←←←←←←←←←←←←←←←←←←←←←←←
             ↖ ↖←↖←←←←←←←←←←←←←←←←←←←←←←←←←←
           ↖            ←←←←←←←←←←←←←←←←←
           ↖       ↖
             ↖ ↖ ↖
''', colors=["#89FAFF","#8BD4FE"])

fog = Gradiant('''
    fff         ▒▒▒
             ▒▒▒▒f▒▒▒▒▒▒▒
   ff   ▒▒▒▒▒ff▒ff▒▒▒▒ff▒▒▒▒▒▒
     ▒▒▒▒▒▒fff▒▒▒▒▒fff▒▒▒▒ffff▒▒▒
           ▒▒▒▒    ▒▒▒▒▒▒
 fff   fff fffff     ffff    ▒▒▒▒▒
             ▒ ▒▒▒▒        ▒▒▒ffff▒▒▒
  ffff   ▒▒▒f▒▒ff▒▒▒▒▒▒      ▒▒▒▒
    ▒▒▒▒▒ff▒ff▒▒▒▒ff▒▒▒▒▒▒▒▒
  ▒▒▒▒▒▒ff▒▒▒▒▒ffff▒▒▒▒ffff▒▒▒▒▒
       ▒▒▒▒ff▒▒▒▒▒▒ fffff  fff f
          ▒▒▒▒
''', colors=["#F5F5F5","#FFFFFF","#F5F5F5"])

icy_fog = Gradiant('''
    ***         ▒▒▒
             ▒▒▒▒*▒▒▒▒▒▒▒
   **   ▒▒▒▒▒++▒++▒▒▒▒++▒▒▒▒▒▒
     ▒▒▒▒▒▒+++▒▒▒▒▒+++▒▒▒▒****▒▒▒
           ▒▒▒▒    ▒▒▒▒▒▒
 ***   *** **++*     ****    ▒▒▒▒▒
  ****             ▒ ▒▒▒▒        ▒▒▒****▒▒▒
     ▒▒▒*▒▒**▒▒▒▒▒▒      ▒▒▒▒
    ▒▒▒▒++▒++▒++▒▒▒▒++▒▒▒▒▒▒▒▒
  ▒▒▒▒▒▒▒▒▒▒▒++++▒▒▒▒++++▒▒▒▒▒
       ▒▒▒▒**▒▒▒▒▒▒ ******  *** *
          ▒▒▒▒
''', colors=["#F5F5F5","#98AFC7","#F5F5F5"])

haze = Gradiant(r'''
 ~          ~         ~
  ``\     /   \     /   \     /``
     \   /     \   /     \   /
       ~         ~         ~
          ~         ~         ~
        /   \     /   \     /   \
       /     \   /     \   /     \
    ~~         ~         ~         ~~
           ~         ~         ~
 ``\     /   \     /   \     /   \
    \   /     \   /     \   /     \
      ~         ~         ~        ~~
''', colors=["#F5F5F5","#98AFC7"])

mist = Gradiant('''
              *           *
   **       *   *       *   *       **
      *   *       *   *       *   *
        *           *           *

   *         *           *        *
        *          *          *       *
            *           *
 **       *   *       *   *       **
    *   *       *   *       *   *
      *           *           *
''', colors=["#F5F5F5","#98AFC7"])


WMO_CODES = {
	0: ('Clear', 'clear'),
	1: ('Mostly Clear', 'mostly-clear'),
	2: ('Partly Cloudy', 'partly-cloudy'),
	3: ('Overcast', 'overcast'),
	5: ('Hazy', 'haze'),
	10: ('Misty', 'mist'),
	45: ('Fog', 'fog'),
	48: ('Icy Fog', 'rime-fog'),
	51: ('Light Drizzle', 'light-drizzle'),
	53: ('Drizzle', 'moderate-drizzle'),
	55: ('Heavy Drizzle', 'dense-drizzle'),
	80: ('Light Showers', 'light-rain'),
	81: ('Showers', 'moderate-rain'),
	82: ('Heavy Showers', 'heavy-rain'),
	61: ('Light Rain', 'light-rain'),
	63: ('Rain', 'moderate-rain'),
	65: ('Heavy Rain', 'heavy-rain'),
	56: ('Light Freezing Drizzle', 'light-freezing-drizzle'),
	57: ('Freezing Drizzle', 'dense-freezing-drizzle'),
	66: ('Light Freezing Rain', 'light-freezing-rain'),
	67: ('Freezing Rain', 'heavy-freezing-rain'),
	71: ('Light Snow', 'slight-snowfall'),
	73: ('Snow', 'moderate-snowfall'),
	75: ('Heavy Snow', 'heavy-snowfall'),
	77: ('Snow Grains', 'snowflake'),
	79: ('Sleet', 'sleet'),
	85: ('Light Snow Showers', 'slight-snowfall'),
	86: ('Snow Showers', 'heavy-snowfall'),
	95: ('Thunderstorm', 'thunderstorm'),
	96: ('Light Thunderstorm with Hail', 'thunderstorm-with-hail'),
	97: ('Thunderstorm with Precipitation', 'thunderstorm-with-rain'),
	99: ('Thunderstorm with Hail', 'thunderstorm-with-hail')
}

STATES = {
    "alabama":"AL",
    "alaska":"AK",
    "arizona":"AZ",
    "arkansas":"AR",
    "california":"CA",
    "colorado":"CO",
    "connecticut":"CT",
    "delaware":"DE",
    "florida":"FL",
    "georgia":"GA",
    "hawaii":"HI",
    "idaho":"ID",
    "illinois":"IL",
    "indiana":"IN",
    "iowa":"IA",
    "kansas":"KS",
    "kentucky":"KY",
    "louisiana":"LA",
    "maine":"ME",
    "maryland":"MD",
    "massachusetts":"MA",
    "michigan":"MI",
    "minnesota":"MN",
    "mississippi":"MS",
    "missouri":"MO",
    "montana":"MT",
    "nebraska":"NE",
    "nevada":"NV",
    "new hampshire":"NH",
    "new jersey":"NJ",
    "new mexico":"NM",
    "new york":"NY",
    "north carolina":"NC",
    "north dakota":"ND",
    "ohio":"OH",
    "oklahoma":"OK",
    "oregon":"OR",
    "pennsylvania":"PA",
    "rhode island":"RI",
    "south carolina":"SC",
    "south dakota":"SD",
    "tennessee":"TN",
    "texas":"TX",
    "utah":"UT",
    "vermont":"VT",
    "virginia":"VA",
    "washington":"WA",
    "west virginia":"WV",
    "wisconsin":"WI",
    "wyoming":"WY",
    "district of columbia":"DC",
    "guam":"GU",
    "marshall islands":"MH",
    "Northern Mariana Island":"MP",
    "puerto rico":"PR",
    "virgin islands":"VI"
}

def get_icon(code, wind_gusts, wind_speed, is_day):

    """Get description and icon from WMO Weather code."""

    desc = "Unable to determine conditions."
    if code in WMO_CODES:
        desc = WMO_CODES[code][0]

    if code == 0:
        if is_day:
            icon = sun_clear
        else:
            icon = moon_clear
    elif code == 1 or code == 2:
        if is_day:
            icon = partly_sunny
        else:
            icon = partly_cloudy
    elif code == 3:
        icon = cloudy
    elif code == 5:
        icon = haze
    elif code == 10:
        icon = mist
    elif code == 45:
        icon = fog
    elif code == 48:
        icon = icy_fog
    elif code in [51,53]:
        icon = light_rain
    elif code in [55,80,81,61,63]:
        icon = rain
    elif code in [65,82]:
        icon = heavy_rain
    elif code in [56,57,66,67,79]:
        icon = icy_rain
    elif code in [71,73,77,85]:
        icon = snow
    elif code in [75, 86]:
        icon = heavy_snow
    elif code == 95:
        icon = thunderstorm
    elif code == 97:
        icon = thunderstorm_rain
    elif code in [96,99]:
        icon = thunderstorm_hail
    elif float(wind_gusts) > 10.0:
        icon = windy
    elif float(wind_speed) > 10.0:
        icon = windy
    else:
        icon = """
        ¯\\_(ツ)_/¯
        """
    return (icon, desc)

def get_wind_rose_direction(degree):

    """Get compass by radial degree."""

    wind_direction = "N"
    if 0 <= degree < 22.5:
        wind_direction =  "N 🢁"
    elif 22.5 <= degree < 45:
        wind_direction =  "NNE"
    elif 45 <= degree < 67.5:
        wind_direction =  "NE 🡼"
    elif 67.5 <= degree < 90:
        wind_direction =  "ENE"
    elif 90 <= degree < 112.5:
        wind_direction =  "East 🡸"
    elif 112.5 <= degree < 135:
        wind_direction =  "ESE"
    elif 135 <= degree < 157.5:
        wind_direction =  "SE 🢇"
    elif 157.5 <= degree < 180:
        wind_direction =  "SSE"
    elif 180 <= degree < 202.5:
        wind_direction =  "South 🡻"
    elif 202.5 <= degree < 225:
        wind_direction =  "SSW"
    elif 225 <= degree < 247.5:
        wind_direction =  "SW 🡾"
    elif 247.5 <= degree < 270:
        wind_direction =  "WSW"
    elif 270 <= degree < 292.5:
        wind_direction =  "West 🡺"
    elif 292.5 <= degree < 315:
        wind_direction =  "WNW"
    elif 315 <= degree < 337.5:
        wind_direction =  "NW 🡽"
    elif 337.5 <= degree < 360:
        wind_direction =  "NNW"
    elif degree == 360:
        wind_direction =  "North"

    return wind_direction

def set_conf():

    """Creates config folder and default config file if they don't exist.
    Folder lives in ~/.config."""

    # Get the user and build the path.
    username = getpass.getuser()
    set_path = "/home/{}/.config/biyori".format(username)
    set_file = "biyori.conf"
    default_conf = os.path.join(set_path, set_file)

    if not os.path.exists(set_path):
        os.mkdir(set_path)

    if not os.path.exists(default_conf):
        config = ConfigParser()
        config['DEFAULT'] = {'location': '40.036282,-83.000762',
                             'state': 'Ohio',
                             'temperature_unit': 'fahrenheit',
                             'wind_speed_unit': 'mph',
                             'precipitation_unit': 'inch',
                             'refresh_rate': '15'
                             }

        with open(default_conf,'w') as conf_file:
            config.write(conf_file)

class BiyoriApp(App):

    """Biyori Weather App for Wave Terminal."""


    # Styling...
    CSS = """
    Screen {
        align: center middle;
    }
    TabbedContent {
        dock: top;
        margin-top: 2;
    }
    TabbedContent {
        height: 90vh;
    }
    TabbedContent #--content-tab-current_tab {
        color: cyan;
    }
    TabbedContent #--content-tab-outlook_tab {
        color: blue;
    }
    TabbedContent #--content-tab-red {
        color: red;
    }
    TabbedContent #--content-tab-magenta {
        color: magenta;
    }
    .grid_tab {
        layout: grid;
        grid-size: 2;
    }
    #current_tab {
        max-width: 100;
    }
    #current {
        margin: 1;
        padding: 1;
        height: 15;
        max-width: 50;
        max-height: 15;
        align: left top;
    }
    Vertical {
        align: left middle;
    }
    #current_bottoms {
        column-span: 2;
        width: 100%;
    }
    #icon {
        height: 15;
        max-width: 50;
        max-height: 15;
        align: left top;
    }
    #bottom {
        column-span: 2;
        height: 5;
        max-height: 5;
        width: 100%;
        align: left middle;
    }
    .box {
        height: 15;
        border: solid cyan;
    }
    #bottom > .sparkline--max-color {
        color: $error;
    }
    #bottom > .sparkline--min-color {
        color: $success;
    }
    """

    # Add/Bind user interface buttons.
    BINDINGS = [
        #("d", "toggle_dark", "Toggle light/dark"),
        ("r", "radar", "Radar (^C Close)"),
        ("u", "refresh", "Update")
    ]

    def __init__(self):
        super().__init__()
        # Set Textual header.
        self.title = "Biyori"
        self.show_clock = True

        # Init config. vars.
        self.daytime = 1
        self.days = 3
        self.lat = 40.036282
        self.long = -83.000762
        self.state_abv = "OH"
        self.temp_unit = "fahrenheit"
        self.wind_unit = "mph"
        self.precip_unit = "inch"
        self.refresh_rate = 15
        self.astro_dt = None

    def get_configs(self) -> None:

        """Reads conf file and pulls relevant configuration data."""

        config = ConfigParser()

        username = getpass.getuser()
        set_path = f"/home/{username}/.config/biyori"
        set_file = "biyori.conf"
        conf_path = os.path.join(set_path, set_file)

        config.read(conf_path)

        sect_default = config["DEFAULT"]
        self.lat, self.long = sect_default["location"].split(',')
        state = sect_default["state"].lower().strip()
        self.state_abv = STATES[state]
        self.temp_unit = sect_default["temperature_unit"].lower().strip()
        self.wind_unit = sect_default["wind_speed_unit"].lower().strip()
        self.precip_unit = sect_default["precipitation_unit"].lower().strip()
        self.refresh_rate = int(sect_default["refresh_rate"])

    # WEATHER REQUESTS
    def get_forcast_uri(self, lat, long):

        """ Fetch the forecasts rest uri. """

        forecast_endpoint = ""
        try:
            # Fetch forecasts url.
            endpoint = f"https://api.weather.gov/points/{lat},{long}"
            response = requests.get(endpoint)

            if response.status_code == 200:

                content = response.json()

                forecast_endpoint = content["properties"]["forecast"]

                response.close()
        except:
            pass

        return forecast_endpoint

    def update_astro(self) -> None:

        """Only update on first run or next day. Hack, I know..."""

        now = datetime.datetime.now()

        if not self.astro_dt:
            self.get_astro(self.lat, self.long)
            self.astro_dt = now

        if now.date() > self.astro_dt.date():
            self.get_astro(self.lat, self.long)
            self.astro_dt = now

    def get_astro(self, lat, long) -> None:

        """Get sun and moon data."""

        tz = tzlocal.get_localzone_name()
        tz_offset = pytz.timezone(tz)
        now = datetime.datetime.now()

        fmt_zone = '%z'
        fmt_date = '%Y-%m-%d'
        localize_tz = tz_offset.localize(now)
        localize_dt = tz_offset.localize(now)

        #urllib.parse.quote()
        str_zone = localize_tz.strftime(fmt_zone)
        str_zone = str_zone.replace('0','')
        str_date = localize_dt.strftime(fmt_date)

        endpoint = f"https://aa.usno.navy.mil/api/rstt/oneday?date={str_date}&coords={lat},{long}&tz={str_zone}&dst=true&id=BiyoriLinuxWeatherTUI"
        response = requests.get(endpoint)
        sun_set = "N/A"
        sun_rise = "N/A"
        moon_phase = "N/A"
        if response.status_code == 200:
            try:
                content = response.json()
                moon_phase = content['properties']['data']['curphase'].lower().strip()
                sun = content['properties']['data']['sundata']
                for sun_data in sun:
                    if sun_data['phen'].lower() == "set":
                        sun_set = sun_data['time'].lower().replace('dt','')
                    if sun_data['phen'].lower() == "rise":
                        sun_rise = sun_data['time'].lower().replace('dt', '')
            except:
                pass
            sun_icon = "☀️"
            moon_icon = "🌙"
            phases ={
                "new moon":"🌑",
                "waning crescent":"🌘",
                "last quarter":"🌗",
                "waning gibbous":"🌖",
                "full moon":"🌕",
                "waxing gibbous":"🌔",
                "first quarter":"🌓",
                "waxing crescent":"🌒"
            }
            str_sun = f" {moon_icon} Sunset: {sun_set} {sun_icon} Sunrise: {sun_rise}"
            moon_phase_icon = phases[moon_phase]
            str_moon_phase = f" {moon_phase_icon} {moon_phase}"
            self.query_one("#sunmoon", Static).update(str_sun+"\n\n"+str_moon_phase)

    def get_current(self,lat,long,temp_unit,wind_unit,precip_unit,alert) -> None:

        """Fetch current weather conditions from Open Meteo. NWS does not provide current.
        Some of these values seem way off... Might be better to use another service."""
        tz = tzlocal.get_localzone_name()
        tz_name = urllib.parse.quote(tz, safe='')
        now = datetime.datetime.now()
        fmt = '%I:%M %p'

        try:
            endpoint = f"https://api.open-meteo.com/v1/gfs?latitude={lat}&longitude={long}&current=rain,showers,snowfall,temperature_2m,relative_humidity_2m,precipitation,cloud_cover,wind_speed_10m,wind_direction_10m,wind_gusts_10m,weather_code,is_day&timezone={tz_name}&forecast_days=1&temperature_unit={temp_unit}&wind_speed_unit={wind_unit}&precipitation_unit={precip_unit}"
            response = requests.get(endpoint)

            if response.status_code == 200:
                content = response.json()

                temperature = content["current"]["temperature_2m"]
                humidity = content["current"]["relative_humidity_2m"]
                precip = content["current"]["precipitation"]
                wind_speed = content["current"]["wind_speed_10m"]
                wind_direction = content["current"]["wind_direction_10m"]
                wind_gusts = content["current"]["wind_direction_10m"]
                cloud_cover = content["current"]["cloud_cover"]
                #rainfall = content["current"]["rain"]
                #showers = content["current"]["showers"]
                #snowfall = content["current"]["snowfall"]
                weather_code = int(content["current"]["weather_code"])
                self.daytime = int(content["current"]["is_day"])

                icon, desc = get_icon(weather_code, wind_gusts, wind_speed, self.daytime)

                wind_rose = get_wind_rose_direction(int(wind_direction))
                t_unit = "F"
                if temp_unit.lower().startswith("cel"):
                    t_unit = "C"

                temp_icon = "🌡️"
                humd_icon = "💧"
                prec_icon = "☔"
                cldc_icon = "🌥️"
                wind_icon = "🪁"
                rose_icon = "🧭"
                curr_icon = "💠"
                alert_icon = "🔔"
                
                weather = Text.from_markup(f"{temp_icon}[bold] Temp: [/bold] [yellow]{temperature}°{t_unit}[/yellow]\n\n{humd_icon} [bold]Humidity:[/bold] [cyan]{humidity}%[/cyan]\n\n{prec_icon} [bold]Precipitation:[/bold] [blue]{precip} {precip_unit}[/blue]\n\n{cldc_icon} [bold]Cloud Cover:[/bold] [green]{cloud_cover}%[/green]\n\n{wind_icon} [bold]Wind Speed:[/bold] [purple]{wind_speed} {wind_unit}[/purple]\n\n{rose_icon} [bold]Direction:[/bold] [red]{wind_rose}[/red]")
                self.query_one("#icon", Static).update(icon)
                self.query_one("#current", Static).update(weather)
                #self.query_one("#bottom", Sparkline).data = [90, 95, 100, 98, 95, 90, 88, 86]
                if alert:
                    desc = f"{desc} {alert_icon}"
                    self.app.bell()

                str_dt_updated = now.strftime(fmt)

                self.query_one("#bottom", Static).update(f" {curr_icon} [bold cyan]Current Conditions ([italic]{str_dt_updated}[/italic]):[/bold cyan] {desc}\n")
            else:
                self.query_one("#current", Static).update("Unable to retrieve weather report.")
            response.close()
        except:
            self.query_one("#current", Static).update("Unable to retrieve weather report.\nCheck your internet connection or Open Medeo's service.")

    def get_forecasts(self, endpoint, forecast_count) -> None:

        """Fetch future forecast data from the National Weather Service."""

        try:
            # Loop forecasts.
            response = requests.get(endpoint)

            if response.status_code == 200:

                forecasts = response.json()

                count = 0
                for period in forecasts["properties"]["periods"]:

                    if count == forecast_count:
                        break

                    name = period["name"]
                    temp = "{} {}".format(period["temperature"], period["temperatureUnit"])
                    if "dewpoint" not in period:
                        dewp = "n/a"
                    else:
                        dewp = "{:.2f} {}".format(period["dewpoint"]["value"],
                                                  period["dewpoint"]["unitCode"].Split(":")[1])

                    if "relativeHumidity" not in period:
                        relhm = "n/a"
                    else:
                        relhm = "{}%".format(period["relativeHumidity"]["value"])

                    wind = "0"
                    direction = "n/a"
                    if period["windSpeed"]:
                        wind = period["windSpeed"]
                        direction = period["windDirection"]

                    # icon = period["icon"] # these are terrible
                    forecast = period["detailedForecast"]

                    forecast_msg = f"[bold]{name}[/bold]\n\nTemp: {temp}\nDewPoint: {dewp}\nRel. Humidity: {relhm}\nWindSpeed: {wind}\nDirection: {direction}\n\n{forecast}\n"
                    if count == 0:
                        self.query_one("#forecast_one", Static).update(Gradiant(forecast_msg))
                    if count == 1:
                        self.query_one("#forecast_two", Static).update(Gradiant(forecast_msg))
                    if count == 2:
                        self.query_one("#forecast_three", Static).update(Gradiant(forecast_msg))
                    count += 1

            response.close()
        except:
            self.query_one("#forecast_one", Static).update("Unable to retrieve weather forecasts.")

    def get_alerts(self, state_code) -> int:

        """Fetch weather alerts from the National Weather Service."""

        alert_msg = ""
        alerts = 0

        try:
            endpoint = f"https://api.weather.gov/alerts/active?area={state_code}"

            response = requests.get(endpoint)

            if response.status_code == 200:

                content = response.json()

                alerts = content["features"]

                if alerts:
                    for alert in alerts:
                        headline = alert["properties"]["headline"]
                        eff_date = alert["properties"]["effective"]
                        end_date = alert["properties"]["expires"]
                        locales = alert["properties"]["areaDesc"]
                        desc = alert["properties"]["description"]
                        alert_msg = f"[bold red]ALERT![/bold red][cyan]\n\n[bold magenta]{headline}[/bold magenta] \n\n[bold yellow]Effective:[/bold yellow] {eff_date}\n[bold yellow]Ends:[/bold yellow] {end_date}\n\n[bold blue]Affects:[/bold blue] {locales}\n\n[bold magenta]Alert:[/bold magenta] {desc}\n[/cyan]"
                        break # return most recent
                    response.close()

                    self.query_one("#alert", Static).update(alert_msg)
                    alerts = 1
                else:
                    self.query_one("#alert", Static).update(" [lime]No alerts for your area.[/lime]")
            else:
                self.query_one("#alert", Static).update(" [bold red]Unable to retrieve alerts.[/bold red]")

            response.close()

        except:
            alerts = 1
            self.query_one("#alert", Static).update("Unable to retrieve alerts.")

        return alerts

    def future_cond(self, lat, long, temp_unit, precip_unit) -> None:
        try:
            tz = tzlocal.get_localzone_name()
            tz_name = urllib.parse.quote(tz, safe='')
            fmt = "%Y-%m-%dT%H:%M"
            now = datetime.datetime.now()

            endpoint = f"https://api.open-meteo.com/v1/forecast?latitude={lat}&longitude={long}&hourly=temperature_2m,relative_humidity_2m,precipitation&timezone={tz_name}&forecast_days=3&temperature_unit={temp_unit}&precipitation_unit={precip_unit}"
            response = requests.get(endpoint)

            if response.status_code == 200:

                content = response.json()

                dts = content['hourly']['time']
                tmp = content['hourly']['temperature_2m']
                hmd = content['hourly']['relative_humidity_2m']
                prc = content['hourly']['precipitation']

                zip_data = zip(dts, tmp, hmd, prc)
                data = []

                count = 0
                for i in zip_data:

                    date_obj = datetime.datetime.strptime(i[0], fmt)
                    if date_obj > now:
                        if count > 3:
                            break
                        #print(date_obj.strftime("%I:%M %p"), f"{tmps[1]}° F")
                        #hrly_temp = (date_obj.strftime("%I:%M %p"), f"{tmps[1]}° F")
                        hrly = (date_obj, date_obj.strftime("%I:%M %p"), i[1], i[2], i[3])
                        data.append(hrly)
                        count += 1

                times=[]
                temps = []
                humid = []
                precp = []
                for i in sorted(data):
                    times.append(i[1])
                    temps.append(i[2])
                    humid.append(i[3])
                    precp.append(i[4])
                pltt = self.query_one("#temps", PlotextPlot).plt
                pltt.bar(times, temps, width = .2)
                pltt.title("Temperature")
                plth = self.query_one("#humidity", PlotextPlot).plt
                plth.bar(times, humid, width = .2)
                plth.title("Rel. Humidity")
                pltp = self.query_one("#precip", PlotextPlot).plt
                pltp.bar(times, precp, width = .2)
                pltp.title("Precipitation")

            response.close()

        except:
            pass

    # Compose the interface.
    def compose(self) -> ComposeResult:

        yield LoadingIndicator()

        yield Header(show_clock=True, icon = '🔅')
        yield Footer()

        # Add the TabbedContent and display widgets
        with TabbedContent(initial="current_tab"):
            with TabPane("Current", id="current_tab", classes="grid_tab"):
                yield Static("", id="current", classes="current_box")
                yield Vertical(Static("", id="icon", classes="current_box"))
                yield Static("", id="bottom", classes="current_bottoms")
                yield Static("", id="sunmoon", classes="current_bottoms")
            with TabPane("Outlook", id="outlook_tab"):
                with VerticalScroll():
                    yield Static(id="forecast_one", classes="box")
                    yield Static(id="forecast_two", classes="box")
                    yield Static(id="forecast_three", classes="box")
            with TabPane("Future", id="magenta"):
                yield PlotextPlot(id="temps")
                yield PlotextPlot(id="humidity")
                yield PlotextPlot(id="precip")
            with TabPane("Alerts", id="red"):
                yield VerticalScroll(Static("", id="alert"))

    def on_mount(self) -> None:

        """Kick off app..."""

        # create base config file on start if not present.
        # File will be created in /home/your_user_name/.config/biyori/biyori.conf
        set_conf()

        # Get weather on open.
        self.update_weather()

        # Kick off the automated update.
        self.set_interval((self.refresh_rate * 60), self.update_weather)

    # Switch tab content.
    def action_show_tab(self, tab: str) -> None:

        """Switch to a new tab."""

        self.get_child_by_type(TabbedContent).active = tab

    # Toggles light and dark mode. Should add this to conf file for persistence.
    '''def action_toggle_dark(self) -> None:

        """An action to toggle dark mode."""

        self.theme = (
            "textual-dark" if self.theme == "textual-light" else "textual-light"
        )'''

    # Open RADAR.
    @work(thread=True, exclusive=True)
    def action_radar(self) -> None:

        """Play the RADAR animation."""
        #You must have the scripts get_radar.py and radar.sh in your Biyori
        # folder and it must be a virtual environment with all the modules installed.
        # Simple path: /home/your_user_name/Biyori/radar.sh
        username = getpass.getuser()
        subprocess.call(["alacritty", "-e", f"/home/{username}/Biyori/radar.sh"], shell=False)

    # Handle refresh button.
    def action_refresh(self) -> None:

        """Called when the input changes"""

        self.update_weather()

    # Weather refresh all.
    @work(thread=True, exclusive=True)
    def update_weather(self) -> None:

        """Get configs and fetch weather data."""

        self.query_one(LoadingIndicator).display = True

        # Get user configuration data.
        # Reload each time to catch any user changes while app is running.
        self.get_configs()

        # Get current weather.
        has_alerts = self.get_alerts(self.state_abv)
        self.get_current(self.lat,self.long,self.temp_unit,self.wind_unit,self.precip_unit,has_alerts)
        #self.get_astro(self.lat,self.long)
        self.update_astro()
        uri = self.get_forcast_uri(self.lat,self.long)
        self.get_forecasts(uri, self.days)
        self.future_cond(self.lat, self.long, self.temp_unit, self.precip_unit)

        self.query_one(LoadingIndicator).display = False

def main():
    app = BiyoriApp()
    app.run()

if __name__ == "__main__":
    main()
